' Penrose Type 2 tilings by deflation
' Rev 1.0 William M Leue, 11/30/2021

option base 1
option angle degrees

' constants
const DEBUG = 0
const NTYPES = 2
const NCORNERS = 4
const G = (1.0 + sqr(5))/2.0 ' yes it's phi
const IG = 1.0/G
const T = 36.0
const MAX_TILES = 40000
const NPARAMS = 5
const MAX_VERTEX_TILES = 5
const ORIG_NUM_TILES = 5
const ISIZE = 140
const NVINFO = 3
const NVERTICES = 7
const MAXVTCNT = 5
const XMID = mm.hres\2
const YMID = mm.vres\2
const ARCOFF = -90.0

' Commands
const UP    = 128
const DOWN  = 129
const LEFT  = 130
const RIGHT = 131
const ENTER = 13
const HOME  = 134
const SPACE = 32
const ESC   = 27

' Tile component indices
const I_X = 1
const I_Y = 2
const I_T = 3
const I_A = 4
const I_S = 5

' Tile Types
const KITE = 1
const DART = 2
const D    = 68

' Color Schemes
const NCSCHEMES   = 4
const DEF_COLOR   = 1
const CS_DEFAULT  = 1
const CS_BLUGRN   = 2
const CS_ROTATION = 3
const CS_WHITE    = 4
const CSCALE      = 15.0

' Vertex Types
const STAR = 1
const ACE = 2
const SUN = 3
const KING = 4
const JACK = 5
const QUEEN = 6
const DEUCE = 7

' Start Screen
const NCATEGORIES = 3
const SIZE_NORMAL = 140.0
const SIZE_LARGE  = 300.0
const DEF_VERTEX  = 3
const DEF_SIZE    = 1

' globals
' VERY IMPORTANT that these variables are float!
dim float tcorners(NTYPES, NCORNERS)
dim float tiles(MAX_TILES, NPARAMS)
dim float dtiles(MAX_TILES, NPARAMS)
dim float size = ISIZE
dim float dsize = ISIZE/G
dim float sat = 1.0
dim float brt = 1.0
' rest of the global variables should be integer
dim integer num_tiles = 0
dim integer num_dtiles = 0
dim integer true_num_tiles = 0
dim integer tcolors(2)
dim integer num_deflations = 0
dim integer start_vertex_type = SUN
dim integer vtcount(NVERTICES)
dim integer vcenter(NVERTICES, 2)
dim integer vinfo(NVERTICES, MAXVTCNT, NVINFO)
dim integer cx, cy
dim vnames$(NVERTICES) = ("Star", "Ace", "Sun", "King", "Jack", "Queen", "Deuce")
dim nitems(NCATEGORIES) = (NVERTICES, 2, 5)
dim start_cylocs(NCATEGORIES) = (50, 100, 150)
dim sizes(2) = (SIZE_NORMAL, SIZE_LARGE)
dim color_schemes(5, 2)
dim cscheme = DEF_COLOR
dim start_tile_counts(NVERTICES, 2)
dim ycenter(NVERTICES) = (YMID, YMID-ISIZE\2, YMID, YMID-ISIZE\3, YMID-10, YMID-ISIZE\3-10, YMID+ISIZE\3)

' Main program
open "debug.txt" for output as #1
ReadTileCorners
ReadVertexInfo
InitColorSchemes
InitTileCounts
cls
if len(mm.Cmdline$) = 0 then
  print "Want to read Help and Info? (Y,N): ";
  input "", a$
  if LEFT$(UCASE$(a$), 1) = "Y" then
    RunProgramWithParams "Penrose_Help", ""
    end
  end if
end if
do
  StartScreen
  cls
  StartTiling start_vertex_type
  HandleEvents
loop
end

sub RunProgramWithParams name$, par$
  local prg$
  prg$ = name$ + ".bas"
  Execute("Run " + chr$(34) + prg$ + chr$(34) + "," + par$)
end sub

' Handle user keyboard inputs
sub HandleEvents
  local z$
  local integer cmd, t0, et
  z$ = INKEY$
  do
    do
      z$ = INKEY$
    loop until z$ <> ""
    cmd = asc(UCASE$(z$))
    select case cmd
      case SPACE
        if num_tiles > MAX_TILES\3 then
          DrawMessage "Too many tiles to deflate again -- Press HOME to start over"
          continue do
        end if
        t0 = timer
        inc num_deflations
        DeflateTiles
        math add dtiles(), 0, tiles()
        num_tiles = num_dtiles
        if num_deflations < 2 then
          true_num_tiles = start_tile_counts(start_vertex_type, num_deflations+1)
        else
          true_num_tiles = 3*true_num_tiles
        end if
        cls
        DrawTiles num_tiles, tiles()
        size = dsize
        dsize = size/G
        et = timer - t0
        DrawTilingLabel et
      case HOME
        exit do
      case ESC
        cls
        close #1
        end
    end select
  loop
end sub

' Read the tile corner locations for Kites and Darts
sub ReadTileCorners
  local i, j
  for i = 1 to NTYPES
    for j = 1 to NCORNERS
      read tcorners(i, j)
    next j
  next i
end sub

' Read the unique vertex info
sub ReadVertexInfo
  local i, j, k
  for i = 1 to NVERTICES
    read vtcount(i)
    for j = 1 to 2
      read vcenter(i, j)
    next j
    for j = 1 to vtcount(i)
      for k = 1 to NVINFO
        read vinfo(i, j, k)
      next k
    next j
  next i
end sub

' Set up the Color Schemes
sub InitColorSchemes
  color_schemes(1, 1) = rgb(255, 155, 0)
  color_schemes(1, 2) = rgb(yellow)
  color_schemes(2, 1) = rgb(green)
  color_schemes(2, 2) = rgb(blue)
  color_schemes(3, 1) = rgb(black)
  color_schemes(3, 2) = rgb(black)
  color_schemes(4, 1) = rgb(white)
  color_schemes(4, 2) = rgb(white)
end sub

' Initialize the tile counts
sub InitTileCounts
  local i
  start_tile_counts(1, 1) = 5 : start_tile_counts(1, 2) = 10
  start_tile_counts(2, 1) = 3 : start_tile_counts(2, 2) = 10
  start_tile_counts(3, 1) = 5 : start_tile_counts(3, 2) = 15
  start_tile_counts(4, 1) = 5 : start_tile_counts(4, 2) = 13
  start_tile_counts(5, 1) = 5 : start_tile_counts(5, 2) = 15
  start_tile_counts(6, 1) = 5 : start_tile_counts(6, 2) = 16
  start_tile_counts(7, 1) = 4 : start_tile_counts(7, 2) = 13
end sub

' Start a tiling with the selected vertex
sub StartTiling v
  local integer j, ncols, nrows
  local float l1, l2, x, y, kx1, ky1, kx2, ky2, a1, a2
  local float tx1, ty1, tx2, ty2, angle
  l1 = size
  l2 = size*G
  cx = mm.hres\2 : cy = ycenter(v)
  zt = vcenter(v, 1)
  zoff = vcenter(v, 2)
  x = cx
  if zoff = 1 then
    y = cy - l1
  else if zoff = 2 then
    y = cy - l2
  else
    y = cy
  end if
  num_dtiles = 0
  num_tiles = 0
  num_deflations = 0
  angle = T*vinfo(v, zt, 2) - 90.0
  AddTile x, y, vinfo(v, zt, 1), angle, size
  for j = 1 to vtcount(v)
    if j <> zt then
      if vinfo(v, j, 3) = 1 then
        angle = T*vinfo(v, j, 2) - 90.0
        AddTile x, y, vinfo(v, j, 1), angle, size
      end if
    end if
  next j
  select case v
    case STAR
    case ACE
      y = cy
      angle = T*vinfo(v, 3, 2) - 90.0
      AddTile x, y+G*size, KITE, angle, size
      angle = T*vinfo(v, 1, 2) - 90.0
      AddTile x, y+G*size, KITE, angle, size
    case SUN
    case KING
      a1 = dtiles(3, I_A)
      kx1 = cx + G*size*cos(a1)
      ky1 = cy - G*size*sin(a1)
      angle = T*vinfo(v, 5, 2) - 90.0
      AddTile kx1, ky1, KITE, angle, size
      a2 = dtiles(2, I_A) - 2*T
      kx2 = cx + G*size*cos(a2)
      ky2 = cy - G*size*sin(a2)
      angle = T*vinfo(v, 1, 2) - 90.0
      AddTile kx2, ky2, KITE, angle, size
    case JACK
      angle = T*vinfo(v, 4, 2) - 90.0
      AddTile x, y-G*size, KITE, angle, size
      a1 = dtiles(2, I_A) - T
      kx1 = cx + G*size*cos(a1)
      ky1 = cy - G*size*sin(a1)
      angle = T*vinfo(v, 5, 2) - 90.0
      AddTile kx1, ky1, DART, angle, size
      a2 = dtiles(1, I_A) + T
      kx2 = cx + G*size*cos(a2)
      ky2 = cy - G*size*sin(a2)
      angle = T*vinfo(v, 3, 2) - 90.0
      AddTile kx2, ky2, DART, angle, size
    case QUEEN
      y = cy
      a1 = 90.0 + 36.0
      kx1 = cx + G*size*cos(a1)
      ky1 = cy - G*size*sin(a1)
      angle = T*vinfo(v, 2, 2) - 90.0
      AddTile kx1, ky1, KITE, angle, size
      a2 = 90.0 - 36.0
      kx2 = cx + G*size*cos(a2)
      ky2 = cy - G*size*sin(a2)
      angle = T*vinfo(v, 1, 2) - 90.0
      AddTile kx2, ky2, KITE, angle, size
      angle = T*vinfo(v, 4, 2) - 90.0
      AddTile x, y+G*size, KITE, angle, size
      angle = T*vinfo(v, 5, 2) - 90.0
      AddTile x, y+G*size, KITE, angle, size
    case DEUCE
      x = cx
      y = cy-G*size
      angle = T*vinfo(v, 4, 2) - 90.0
      AddTile x, y, DART, angle, size
      a1 = 198
      kx1 = cx + G*size*cos(a1)
      ky1 = cy - G*size*sin(a1)
      angle = T*vinfo(v, 2, 2) - 90.0
      AddTile kx1, ky1, KITE, angle, size
      a2 = -18
      kx2 = cx + G*size*cos(a2)
      ky2 = cy - G*size*sin(a2)
      angle = T*vinfo(v, 1, 2) - 90.0
      AddTile kx2, ky2, KITE, angle, size
  end select
  math add dtiles(), 0, tiles()
  num_tiles = num_dtiles
  true_num_tiles = num_tiles
  DrawTiles num_tiles, tiles()
  DrawTilingLabel 0
end sub

' Draw a Tiling
sub DrawTiles num, ltiles()
  local integer i, j, k, type, ec, fc, r, g, b
  local float x, y, size, angle, tangle, st, hue, hstart
  local xv(NCORNERS), yv(NCORNERS)
  ec = rgb(gray)
  for i = 1 to num
    x = ltiles(i, I_X)
    y = ltiles(i, I_Y)
    type = ltiles(i, I_T)
    angle = ltiles(i, I_A)
    size = ltiles(i, I_S)
    if type = KITE then
      hstart = 0.0
    else
      hstart = 180.0
    end if
    if cscheme = CS_ROTATION then
      cangle = angle
      if cangle < 0.0 then inc cangle, 360.0
      if cangle < 0.0 then inc cangle, 360.0
      if cangle < 0.0 then inc cangle, 360.0
      if cangle > 360.0 then inc cangle, -360.0
      if cangle > 360.0 then inc cangle, -360.0
      if cangle > 360.0 then inc cangle, -360.0
      hue = hstart + CSCALE*cangle/36.0
      HSV2RGB hue, sat, brt, r, g, b
      fc = rgb(r, g, b)
    else
      fc = tcolors(type)
    end if
    tangle = angle - T
    ' inner loop unrolled for speed
    ' saves about 7% of runtime
      xv(1) = x : yv(1) = y
      st = tcorners(type, 2)*size
      xv(2) = x + st*cos(tangle)
      yv(2) = y - st*sin(tangle)
      inc tangle, T
      st = tcorners(type, 3)*size
      xv(3) = x + st*cos(tangle)
      yv(3) = y - st*sin(tangle)
      inc tangle, T
      st = tcorners(type, 4)*size
      xv(4) = x + st*cos(tangle)
      yv(4) = y - st*sin(tangle)
    ' end unrolled inner loop
    polygon NCORNERS, xv(), yv(), ec, fc
    if cscheme = CONWAY then DrawConwayLoops i, ltiles()
  next i
end sub

' Draw a descriptive label on a tiling
sub DrawTilingLabel et
  local t$
  if num_deflations = 0 then
    t$ = "Original Tiling: number of tiles: " + str$(true_num_tiles)
  else
    t$ = "Tiling after " + str$(num_deflations)
    cat t$, " deflations: number of tiles: " + str$(true_num_tiles)
  end if
  DrawMessage t$
  t$ = "Elapsed time: " + str$(et/1000.0) + " secs."
  text mm.hres, 0, t$, "RT"
end sub

' Deflate a tiling to make more tiles (iterative version)
sub DeflateTiles
  local i, j, type
  local sign
  local float x, y, dx, dy, dsize, a, b
  local float ats4, ats1, gsize, ts
  num_dtiles = 0
  dsize = size/G
  gsize = size*G
  for i = 1 to num_tiles
    x = tiles(i, I_X)
    y = tiles(i, I_Y)
    type = tiles(i, I_T)
    angle = tiles(i, I_A)
    xsize = tiles(i, I_S)
    if type = DART then
      sign = 1
      a = angle + 5*T*sign
      AddTile x, y, KITE, a, dsize
      ' unroll inner loop for speed
        ats4 = angle - 4*T*sign
        dx = x + cos(ats4) * gsize
        dy = y - sin(ats4) * gsize
        AddTile dx, dy, DART, ats4, dsize
        sign = -1
        ats4 = angle - 4*T*sign
        dx = x + cos(ats4) * gsize
        dy = y - sin(ats4) * gsize
        AddTile dx, dy, DART, ats4, dsize
      ' end of unrolled inner loop
    else
      sign = 1
      ' inner loop unrolled for speed
        ts = T*sign
        a = angle - 4*ts
        b = angle + 3*ts
        ats1 = angle - ts
        AddTile x, y, DART, a, dsize
        dx = x + cos(ats1) * gsize
        dy = y - sin(ats1) * gsize
        AddTile dx, dy, KITE, b, dsize
        sign = -1
        ts = T*sign
        a = angle - 4*ts
        b = angle + 3*ts
        ats1 = angle - ts
        AddTile x, y, DART, a, dsize
        dx = x + cos(ats1) * gsize
        dy = y - sin(ats1) * gsize
        AddTile dx, dy, KITE, b, dsize
      ' end of unrolled inner loop
    end if    
  next i
end sub

' Add a Tile to the tiling.
sub AddTile x, y, type, angle, tsize
  inc num_dtiles
  dtiles(num_dtiles, I_X) = x
  dtiles(num_dtiles, I_Y) = y
  dtiles(num_dtiles, I_T) = int(type)
  dtiles(num_dtiles, I_A) = int(angle)
  dtiles(num_dtiles, I_S) = tsize
end sub

' Display the Start Screen
sub StartScreen
  local z$, cmd, category, item, i
  local cur_item(NCATEGORIES)
  category = 1
  cur_item(1) = DEF_VERTEX
  cur_item(2) = DEF_SIZE
  cur_item(3) = DEF_COLOR
  DrawStartScreen
  HiliteChoice category, cur_item()
  z$ = INKEY$
  do
    do
      z$ = INKEY$
    loop until z$ <> ""
    cmd = asc(UCASE$(z$))
    select case cmd
      case UP
        inc category, -1
        if category < 1 then category = NCATEGORIES
        item = DEF_VERTEX
        HiliteChoice category, cur_item()
      case DOWN
        inc category
        if category > NCATEGORIES then category = 1
        HiliteChoice category, cur_item()
      case LEFT
        item = cur_item(category)
        inc item, -1
        if item < 1 then item = nitems(category)
        cur_item(category) = item
        HiliteChoice category, cur_item()
      case RIGHT
        item = cur_item(category)
        inc item
        if item > nitems(category) then item = 1
        cur_item(category) = item
        HiliteChoice category, cur_item()
      case ENTER
        item = cur_item(1)
        start_vertex_type = item
        item = cur_item(2)
        size = sizes(item)
        dsize = size/G
        item = cur_item(3)
        cscheme = item
        tcolors(1) = color_schemes(item, 1)
        tcolors(2) = color_schemes(item, 2)
        exit do        
      case ESC
        cls
        end
    end select
  loop
end sub

' Draw the Start Screen
sub DrawStartScreen
  cls
  text mm.hres\2, 1, "Penrose P2 Tilings Setup", "CT", 4,, rgb(green)
  text 40, start_cylocs(1), "Starting Vertex", "LT", 4,, rgb(cyan)
  text 60, start_cylocs(1)+20, "Star    Ace     Sun     King    Jack    Queen   Deuce"
  text 40, start_cylocs(2),    "Tile Size", "LT", 4,, rgb(cyan)
  text 60, start_cylocs(2)+20, "Normal  Large"
  text 40, start_cylocs(3), "Color Scheme", "LT", 4,, rgb(cyan)
  text 60, start_cylocs(3)+20, "1       2       3       4"
  print @(0, 200) ""
  print "Use the UP and DOWN keyboard keys to select a category."
  print "Use the LEFT and RIGHT keyboard keys to select a choice in the category."
  print "Press ENTER to start tiling."
  print ""
  print "NORMAL size keeps all tiles visible at every deflation level."
  print "LARGE size makes tiles clearer at the highest deflation level."
  print ""
  print "Color scheme 1 is orange/yellow."
  print "Color scheme 2 is blue/green."
  print "Color scheme 3 is saturated color depending on tile rotation."
  print "Color scheme 4 is black outlines with white fill color."

  print ""
  print "During tiling:"
  print "  Press the SPACE bar to deflate to the next level. (up to 7 levels)"
  print "  Press the HOME key to return to this menu."
  print "  Press the ESCAPE key to quit the program."
end sub

' Hilight the current start screen choice
sub HiliteChoice category, cur_items()
  local ax, ay, ix, iy
  local xv(3), yv(3)
  DrawStartScreen
  ax = 28
  ay = start_cylocs(category) + 4
  xv(1) = ax      : yv(1) = ay
  xv(2) = ax-10   : yv(2) = ay-5
  xv(3) = ax-10   : yv(3) = ay+5
  polygon 3, xv(), yv(), rgb(red), rgb(red)
  ix = 56 + (cur_items(category)-1)*64
  iy = start_cylocs(category) + 18
  box ix, iy, 55, 16,, rgb(yellow)
end sub

sub DrawMessage m$
  text 10, 580, space$(80)
  text 10, 580, m$
end sub

' Convert an HSV value to its RGB equivalent
' The S and V values must be in range 0..1; the H value must
' be in range 0..360. The RGB values will be in range 0..255.
sub HSV2RGB h as float, s as float, v as float, r as integer, g as integer, b as integer
  local float i, hh, f, p, q, t, x, c, rp, gp, bp
  c = v*s
  hh = h/60.0
  i = int(hh)
  f = hh - i
  p = v*(1-s)
  q = v*(1-s*f)
  t = v*(1-s*(1-f))
  x = c*(1.0 - hh MOD 2 - 1)
  
  select case i
    case 0
      rp = v : gp = t : bp = p
    case 1
      rp = q : gp = v : bp = p
    case 2
      rp = p : gp = v : bp = t
    case 3
      rp = p : gp = q : bp = v
    case 4
      rp = t : gp = p : bp = v
    case 5
      rp = v : gp = p : bp = q
  end select
  r = rp*255.0 : g = gp*255.0 : b = bp*255.0
end sub

' tile vertex locations for Kites and Darts
data 0, G, G, G, 0, -G, -1, -G

' 5 unique filled vertices data:
' first line: number of tiles to fill vertex
' next lines: tile type (KITE (1), DART (2)), rotation (0..9), and corner that
' attaches to vertex (1..4)

' Star
data 5
data 1, 0
data DART, 1, 1
data DART, 3, 1
data DART, 5, 1
data DART, 7, 1
data DART, 9, 1

' Ace
data 3
data 2, 1
data KITE, 6, 2
data DART, 5, 3
data KITE, 4, 4

' Sun
data 5
data 1, 0
data KITE, 0, 1
data KITE, 2, 1
data KITE, 4, 1
data KITE, 6, 1
data KITE, 8, 1

' King
data 5
data 2, 0
data KITE, 2, 4
data DART, 8, 1
data DART, 0, 1
data DART, 2, 1
data KITE, 8, 2

' Jack
data 5
data 1, 0
data KITE, 1, 1
data KITE, 9, 1
data DART, 1, 4
data KITE, 0, 3
data DART, 9, 2

' Queen
data 5
data 3, 0
data KITE, 0, 2
data KITE, 0, 4
data DART, 0, 1
data KITE, 6, 2
data KITE, 4, 4

' Deuce
data 4
data 3, 2
data KITE, 7, 3
data KITE, 3, 3
data DART, 4, 2
data DART, 6, 1
